<?php

namespace App\Services;

use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Stichoza\GoogleTranslate\GoogleTranslate;

class TranslateService
{
    protected GoogleTranslate $translator;

    public function __construct()
    {
        $this->translator = new GoogleTranslate();
        // Auto-detect source language
        $this->translator->setSource(); // null => auto
    }

    /**
     * Translate a collection/array/paginator of models in place.
     * $fields may be a string or array of attributes to translate.
     * HTML-rich fields ('description', 'content', 'body') are processed safely.
     */
    public function translateModelsBatchCached($models, $fields, string $targetLanguage)
    {
        $fields = (array) $fields;

        // Normalize iterable input
        if ($models instanceof LengthAwarePaginator) {
            $iterable = $models->getCollection();
        } elseif ($models instanceof Collection || is_array($models) || $models instanceof \Traversable) {
            $iterable = $models;
        } else {
            $iterable = [$models];
        }

        foreach ($iterable as $model) {
            foreach ($fields as $field) {
                // Handle 'description' vs 'Description'
                $val = $model->getAttribute($field);
                if ($val === null && strtolower($field) === 'description') {
                    $val = $model->getAttribute('Description');
                    if ($val !== null) { $field = 'Description'; }
                }

                if ($val !== null && is_string($val) && $val !== '') {
                    $isHtmlField = in_array(strtolower($field), ['description', 'content', 'body'], true);

                    $model->$field = $isHtmlField
                        ? $this->translateHtmlPreserveTags($val, $targetLanguage)   // translate text nodes only
                        : $this->translatePlainTextCached($val, $targetLanguage);   // plain text
                }
            }
        }

        return $models;
    }

    /**
     * Translate plain text with cache.
     * (No HTML here; just text strings.)
     */
    public function translatePlainTextCached(string $text, string $targetLanguage): string
    {
        // Normalize to something stable for caching
        $payload = html_entity_decode($text, ENT_QUOTES | ENT_HTML5, 'UTF-8');

        // If it's blank after trim, return as-is
        if (trim($payload) === '') {
            return $text;
        }

        $cacheKey = 'translation_text_' . md5($payload . '|' . $targetLanguage);

        return Cache::rememberForever($cacheKey, function () use ($payload, $targetLanguage) {
            $this->translator->setTarget($targetLanguage);
            return $this->translator->translate($payload);
        });
    }

    /**
     * Translate only the inner text of HTML, preserving ALL tags & attributes (style/class/... unchanged).
     * This tokenizer approach avoids DOM rewriting and keeps your markup verbatim.
     */
    public function translateHtmlPreserveTags(string $html, string $targetLanguage): string
    {
        // Split into [TAG][TEXT][TAG][TEXT]... while keeping the tags in the result
        $tokens = preg_split('/(<[^>]+>)/u', $html, -1, PREG_SPLIT_DELIM_CAPTURE);

        if ($tokens === false) {
            // Fallback: if splitting fails, just return original
            return $html;
        }

        $result = '';
        $inScript = false;
        $inStyle  = false;

        foreach ($tokens as $token) {
            if ($token === '') {
                continue;
            }

            if ($this->isTag($token)) {
                // Track whether we are inside <script> or <style>
                if ($this->isOpeningTag($token, 'script')) $inScript = true;
                if ($this->isClosingTag($token, 'script')) $inScript = false;
                if ($this->isOpeningTag($token, 'style'))  $inStyle  = true;
                if ($this->isClosingTag($token, 'style'))  $inStyle  = false;

                // Append tag verbatim (attributes/styles untouched)
                $result .= $token;
                continue;
            }

            // TEXT node (between tags)
            if ($inScript || $inStyle) {
                // Do NOT translate script/style content
                $result .= $token;
                continue;
            }

            // Preserve leading/trailing whitespace; translate only the core
            if (trim($token) === '') {
                $result .= $token;
                continue;
            }

            // Capture leading/trailing whitespace to re-apply after translation
            preg_match('/^\s*/u', $token, $m1);
            preg_match('/\s*$/u', $token, $m2);
            $lead  = $m1[0] ?? '';
            $trail = $m2[0] ?? '';
            $core  = trim($token);

            $translatedCore = $this->translatePlainTextCached($core, $targetLanguage);
            $result .= $lead . $translatedCore . $trail;
        }

        return $result;
    }

    /* ------------ helpers ------------ */

    private function isTag(string $token): bool
    {
        // Starts with '<' and ends with '>'
        return isset($token[0]) && $token[0] === '<' && substr($token, -1) === '>';
    }

    private function isOpeningTag(string $tag, string $name): bool
    {
        return (bool) preg_match('/^<\s*' . preg_quote($name, '/') . '\b[^>]*>$/iu', $tag);
    }

    private function isClosingTag(string $tag, string $name): bool
    {
        return (bool) preg_match('/^<\s*\/\s*' . preg_quote($name, '/') . '\s*>$/iu', $tag);
    }
}
